//*********************************************************************
// Example1 is a bounded, unconstrained optimization problem with 
// four variables. 
// 
// The goal of the optimization is to minimize the following expression   
//  100*(x[2]-x[1]^2)^2 + (1-x[1])^2 + 90*(x[4]-x[3]^2)^2 +
//           (1-x[3])^2 + 10.1*((x[2]-1)^2 + (x[4]-1)^2) + 19.8*(x[2]-1)(x[4]-1)
//
//	The problem has 4 variables.  Each variable may take on values between
// -10.0 and 10.0
//
// The OptQuest.dll distributed with this example is a demo
// version with the following restrictions:
//		- Maximum variables = 7	
//		- Maximum constraints = 4
//		- Maximum requirements =  4
//		- Maximum iterations = 500
//
//*********************************************************************

#include "ocl.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define NUM_VARIABLES 4

double evaluate(double *);

int main(int argc,char *argv[])
{
	double DecVar[NUM_VARIABLES+1];
	double ObjVal, Boundary;
	long   Example, nsol, status;
	int    i, TotalIter;

	if (argc != 3) 
	{
		// If no inputs were given, use these defaults
		TotalIter = 500;
		Boundary = 0.5;
	}
	else
	{
		TotalIter = atoi(argv[1]);
		Boundary  = atof(argv[2]);
	}
	if (TotalIter > 500)
	{
		printf("Warning:  The Demo version is limited to 500 iterations\n");
		TotalIter = 500;
	}

	/* Allocating memory */
	Example = OCLSetup(NUM_VARIABLES,0,0,0,"MIN", 999999999);
	if (Example < 0) 
	{
		printf("OCLSetup error code %d\n", Example);
		exit(1);
	}

	/* Defining variables */
	for (i = 1; i <= NUM_VARIABLES; i++) 
	{
		status = OCLDefineVar(Example, i, -10, OCLGetNull(Example), 10,"CON", 1);
		if (status < 0) {
			printf("OCLDefineVar error code %d\n", status);
			exit(1);
		}
	}
	
	/* Set parameters and initialize the population */
	status = OCLSetSolutions(Example, TotalIter);
	if (status < 0) 
	{
		printf("OCLSetSolutions error code %d\n", status);
		exit(1);
	}

	/* Generate and evaluate TotalIter solutions */
	for (i = 1; i <= TotalIter; i++)
	{
		nsol = OCLGetSolution(Example,DecVar);
		if (nsol < 0) 
		{
			printf("OCLGetSolution error code %d\n", nsol);
			exit(1);
		}
		ObjVal = evaluate(DecVar);
		status = OCLPutSolution(Example,nsol,&ObjVal,(double *)OCLGetNull(Example));
		if (status < 0) 
		{
			printf("OCLPutSolution error code %d\n", status);
			exit(1);
		}
		if (!(i%100)) 
		{
			status = OCLGetBest(Example,DecVar,&ObjVal);
			if (status < 0) {
				printf("OCLGetBest error code %d\n", status);
				exit(1);
			}
			printf("Best solution after %7d iterations is %9.6f\n",i,ObjVal);
		}
	}

	/* Display the best solution found */
	status = OCLGetBest(Example,DecVar,&ObjVal);
	if (status < 0) 
	{
		printf("OCLGetBest error code %d\n", status);
		exit(1);
	}
	printf("\n");
	for(i=1;i<=NUM_VARIABLES;i++) printf("x[%2d] = %9.6lf\n",i, DecVar[i]);

	printf("\nPress any key to end ...\n");
	getchar();
	
	/* Free memory */
	status = OCLGoodBye(Example);
	if (status < 0) {
		printf("OCLGoodBye error code %d\n", status);
		exit(1);
	}
	return 0;
}

/* Evaluation function */

double evaluate(double *x)
{
	return(100*pow(x[2]-pow(x[1],2),2)+pow(1-x[1],2)+90*pow(x[4]-pow(x[3],2),2)+
		pow(1-x[3],2)+10.1*(pow(x[2]-1,2)+pow(x[4]-1,2))+19.8*(x[2]-1)*(x[4]-1));
}
